#!/bin/bash

# Change to any desired directory.
installDir=~

#----------------------------------------
# Check if the "tar" command exists
which tar

# Check the exit status
if [ $? -ne 0 ]; then
    echo "tar command does not exist."
    exit 1
fi

#----------------------------------------
# Check if the "unzip" command exists
which unzip

# Check the exit status
if [ $? -ne 0 ]; then
    echo "unzip command does not exist."
    echo "see https://www.tecmint.com/install-zip-and-unzip-in-linux/"
    exit 1
fi

#----------------------------------------
# Check if the "curl" command exists
which curl

# Check the exit status
if [ $? -ne 0 ]; then
    echo "curl command does not exist."
    echo "to install, sudo apt-get install curl"
    exit 1
fi

#----------------------------------------
# Check if the "sha256sum" command exists
which sha256sum

# Check the exit status
if [ $? -ne 0 ]; then
    echo "sha256sum command does not exist."
    exit 1
fi

# ------------------------------
echo "Installing to $installDir"

# Create the install directory if needed.
if [ ! -d "$installDir" ]; then
	if mkdir $installDir; then
	    echo "Created $installDir"
	else
	    echo "Failed to create $installDir"
 	    exit 1
	fi
fi
cd $installDir

# -------------------------------------------------------------------------
# Create the "chilkatsoft.com" directory underneath the install directory.
if [ ! -d "chilkatsoft.com" ]; then
	if mkdir chilkatsoft.com; then
	    echo "Created chilkatsoft.com"
	else
	    echo "Failed to create chilkatsoft.com"
 	    exit 1
	fi
fi

cd chilkatsoft.com

# --------------------------------------------------------------------
# The native "C" library download is based on the machine architecture
# Download and verify the SHA256 digest of the download is a expected.

# Possible machine architectures are:
# armv7l
# aarch64
# x86_64
# i686
# If your machine architecture is different, please let Chilkat know.
machineArch=`uname -m`

if [ "$machineArch" = "x86_64" ]; then
	echo "x86_64"
	archSubdir=linux-x64-gcc
	curl https://chilkatdownload.com/11.2.0/chilkatext-linux-x64-gcc.tar.gz -o chilkat_native_c.tar.gz
	if [ $? -ne 0 ]; then
		echo "curl download of the Chilkat native "C" library failed."
		exit 1
	fi
	digestSha256=`sha256sum chilkat_native_c.tar.gz`
	echo $digestSha256
	if [[ $digestSha256 == *07e9907ef1d68430f7a305f8de5634ef38ba4ef43658981b169451695cb655ab* ]]; then
		echo "Good, the SHA256 digest of chilkat_native_c.tar.gz is as expected."
	else
		echo "Digest of Chilkat native C library is not the expected value."
		exit 1
	fi
elif [ "$machineArch" = "i686" ]; then
	echo "i686"
	archSubdir=linux-x86-gcc
	curl https://chilkatdownload.com/11.2.0/chilkatext-linux-x86-gcc.tar.gz -o chilkat_native_c.tar.gz
	if [ $? -ne 0 ]; then
		echo "curl download of the Chilkat native "C" library failed."
		exit 1
	fi
	digestSha256=`sha256sum chilkat_native_c.tar.gz`
	echo $digestSha256
	if [[ $digestSha256 == *b7c44fcb8af97277d7d6da8ff49f993b9041976fbdee695f7420b156be0c7bea* ]]; then
		echo "Good, the SHA256 digest of chilkat_native_c.tar.gz is as expected."
	else
		echo "Digest of Chilkat native C library is not the expected value."
		exit 1
	fi
elif [ "$machineArch" = "aarch64" ]; then
	echo "aarch64"
	archSubdir=linux-aarch64-gcc
	curl https://chilkatdownload.com/11.2.0/chilkatext-linux-aarch64-gcc.tar.gz -o chilkat_native_c.tar.gz
	if [ $? -ne 0 ]; then
		echo "curl download of the Chilkat native "C" library failed."
		exit 1
	fi
	digestSha256=`sha256sum chilkat_native_c.tar.gz`
	echo $digestSha256
	if [[ $digestSha256 == *81b45f2f2fc2479483609f9e44705d58a7d05fb3878c66b501e4ff4aa36cc1a2* ]]; then
		echo "Good, the SHA256 digest of chilkat_native_c.tar.gz is as expected."
	else
		echo "Digest of Chilkat native C library is not the expected value."
		exit 1
	fi
elif [ "$machineArch" = "armv7l" ]; then
	echo "armv7l"
	archSubdir=linux-armhf-clang
	curl https://chilkatdownload.com/11.2.0/chilkatext-linux-armhf-clang.tar.gz -o chilkat_native_c.tar.gz
	if [ $? -ne 0 ]; then
		echo "curl download of the Chilkat native "C" library failed."
		exit 1
	fi
	digestSha256=`sha256sum chilkat_native_c.tar.gz`
	echo $digestSha256
	if [[ $digestSha256 == *3950c522d1d154706ae6f64b98a05e9d5f153e320af0c56812355610c10f977d* ]]; then
		echo "Good, the SHA256 digest of chilkat_native_c.tar.gz is as expected."
	else
		echo "Digest of Chilkat native C library is not the expected value."
		exit 1
	fi
else
	echo "Unsupported architecture: $machineArch"
	exit 1
fi


# ---------------------------------------------------------
# Extract and rename the subdirectory to "native_c_lib"
#
# the resulting Chilkat native "C" static library is $installDir/chilkatsoft.com/native_c_lib/libchilkatext.a

if [ -d "native_c_lib" ]; then
	rm -rf native_c_lib
fi

if [ -d "$archSubdir" ]; then
	rm -rf $archSubdir
fi

tar xvf chilkat_native_c.tar.gz
if [ $? -ne 0 ]; then
	echo "extracting from .tar.gz failed."
	exit 1
fi

mv $archSubdir native_c_lib
rm -f chilkat_native_c.tar.gz

# --------------------------------------------------------
# Delete the "chilkat", "chilkat_example1", and "chilkat_example2" subdirectories
# from previous install attempts if they exist.

# (at this point, we are in the $installDir/chilkatsoft.com directory)
if [ -d "chilkat" ]; then
	rm -rf chilkat
fi
if [ -d "chilkat_example1" ]; then
	rm -rf chilkat_example1
fi
if [ -d "chilkat_example2" ]; then
	rm -rf chilkat_example2
fi
# also check for any previously downloaded "license.pdf"
if [ -e "license.pdf" ]; then
	rm -f license.pdf
fi

# --------------------------------------------------------
# Download the chilkat_golang.zip containing the Go source files that implement the Chilkat module, 
# which do so by calling into the Chilkat native "C" lib.

curl https://chilkatdownload.com/11.2.0/chilkat_golang.zip -o chilkat_golang.zip
if [ $? -ne 0 ]; then
	echo "curl download of chilkat_golang.zip failed."
	exit 1
fi

# Verify the SHA256 digest is correct.
digestSha256=`sha256sum chilkat_golang.zip`
echo $digestSha256
if [[ $digestSha256 == *78d39a0d309395c85b1a096793d977b38136a88b5b752e0e938cf84e0c750bc1* ]]; then
	echo "Good, the SHA256 digest of chilkat_golang.zip is as expected."
else
	echo "Digest of chilkat_golang.zip is not the expected value."
	exit 1
fi

# Unzip to create three subdirectories under $installDir/chilkatsoft.com:
# chilkat
# chilkat_example1
# chilkat_example2

unzip -q chilkat_golang.zip
if [ $? -ne 0 ]; then
	echo "unzipping chilkat_golang.zip failed."
	exit 1
fi

rm -f chilkat_golang.zip

# --------------------------------------------------------
echo "Let's build the Chilkat module..."
echo "This can take a minute or two, or three..."

CGO_LDFLAGS="-L$installDir/chilkatsoft.com/native_c_lib -lchilkatext -lresolv -lpthread -ldl -lstdc++"

cd chilkat
go mod init chilkatsoft.com/chilkat
go build
if [ $? -ne 0 ]; then
	echo "go build failed."
	exit 1
fi

echo "Successfully built the Chilkat Go module."




