﻿// The 'using' statements are organized alphabetically for consistency.
using Chilkat;
using Microsoft.Web.WebView2.Core;
using Microsoft.Web.WebView2.WinForms;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using static System.Windows.Forms.VisualStyles.VisualStyleElement;
using TextBox = System.Windows.Forms.TextBox;

namespace TestAi
{
    /// <summary>
    /// Main form for the AI Chat application.
    /// This application uses the Chilkat AI library to interact with various AI providers
    /// and displays the conversation in a WebView2 control.
    /// </summary>
    public partial class Form1 : Form
    {
        #region Private Fields

        /// <summary>
        /// The WebView2 control used to render and display the HTML chat conversation.
        /// </summary>
        private WebView2 _webView;

        /// <summary>
        /// The main Chilkat AI component for handling API requests.
        /// </summary>
        private Chilkat.Ai _aiClient = new Chilkat.Ai();

        /// <summary>
        /// JSON configuration for converting streaming Markdown fragments into JavaScript function calls.
        /// This allows for real-time updates in the WebView2 control.
        /// </summary>
        private Chilkat.JsonObject _streamingOptions = new Chilkat.JsonObject();

        // StringBuilders used to process streaming AI events to avoid repeated string allocations.
        private Chilkat.StringBuilder _sbEventName = new Chilkat.StringBuilder();
        private Chilkat.StringBuilder _sbDelta = new Chilkat.StringBuilder();
        private Chilkat.StringBuilder _sbJsCalls = new Chilkat.StringBuilder();
        private Chilkat.StringBuilder _sbMarkdown = new Chilkat.StringBuilder();
        private Chilkat.StringBuilder _sbAccumulatedMarkdown = new Chilkat.StringBuilder();

        /// <summary>
        /// A cancellation token source to gracefully stop the background polling for AI responses.
        /// </summary>
        private CancellationTokenSource _cancellationTokenSource;

        /// <summary>
        /// Flag to indicate whether a new HTML shell is needed for the conversation.
        /// This is set to true when clearing the conversation or on first load.
        /// </summary>
        private static bool _isNewConversationNeeded = true;

        #endregion

        #region Constructor and Initialization

        public Form1()
        {
            InitializeComponent();
            InitializeSecrets();
            InitializeChilkat();
            InitializeWebView();
            InitializeUI();
        }

        /// <summary>
        /// Loads API keys and the Chilkat unlock code from the system's Secrets Manager, if available.
        /// </summary>
        private void InitializeSecrets()
        {
            CheckLoadSecrets();
        }

        /// <summary>
        /// Unlocks the Chilkat library using the provided code or enters a 30-day trial mode.
        /// </summary>
        private void InitializeChilkat()
        {
            // If no unlock code is present, use a placeholder for the 30-day trial.
            if (string.IsNullOrWhiteSpace(textBoxUnlockCode.Text))
            {
                textBoxUnlockCode.Text = "TRL30D.CB1122025_VNXJUR6cvgw96JwCXAi9+ROEzXAAz33OlerY22vi.8ndvtqkrZvZg5CKnZ9MvF85mRuZZdhBPTGqZMybl9PAItY48t4VZR==";
            }
            UnlockChilkat(textBoxUnlockCode.Text.Trim(),false);

            // Configure options for streaming Markdown-to-HTML conversion.
            // These options tell Chilkat to output JavaScript function calls to update the DOM dynamically.
            _streamingOptions.UpdateBool("streaming", true);
            _streamingOptions.UpdateBool("emitJavascript", true);
        }

        /// <summary>
        /// Initializes the WebView2 control and adds it to the form.
        /// </summary>
        private void InitializeWebView()
        {
            _webView = new WebView2
            {
                Dock = DockStyle.Fill
            };

            tabChat.Controls.Add(_webView);

            // Asynchronously initialize the CoreWebView2.
            _webView.CoreWebView2InitializationCompleted += WebView_CoreWebView2InitializationCompleted;
            _webView.EnsureCoreWebView2Async(null);
        }

        /// <summary>
        /// Sets up the initial state of the user interface controls.
        /// </summary>
        private void InitializeUI()
        {
            // Set default selections for UI dropdowns.
            cmbPrism.Text = "tomorrow";
            cmbHtmlStyles.Text = "ChatGPT";

            // Set the default provider, which will also trigger populating the models list.
            UpdateProvider("");

            // Provide a default example prompt.
            txtInput.Text = "Show me how to write strcpy in C code.";

            // Start the application on the "Secrets" tab for initial setup.
            tabControl1.SelectedIndex = 3;


            treeViewJson.Font = new Font("Consolas", 10);

            treeViewJson.HideSelection = false;
            treeViewJson.ShowLines = true;
            treeViewJson.ShowPlusMinus = true;
            treeViewJson.ShowRootLines = true;

            treeViewJson.FullRowSelect = true;
            treeViewJson.HotTracking = true;
            treeViewJson.Indent = 22;

            treeViewJson.BorderStyle = BorderStyle.FixedSingle;
            treeViewJson.Scrollable = true;

            // Dark theme colors (comment out if you want light mode)
            //treeViewJson.BackColor = Color.FromArgb(32, 32, 32);
            //treeViewJson.ForeColor = Color.Gainsboro;
            //treeViewJson.LineColor = Color.FromArgb(60, 60, 60);

        }

        #endregion

        #region Event Handlers

        /// <summary>
        /// Handles the completion of the WebView2 initialization.
        /// Displays an initial message in the control.
        /// </summary>
        private void WebView_CoreWebView2InitializationCompleted(object sender, CoreWebView2InitializationCompletedEventArgs e)
        {
            if (!e.IsSuccess)
            {
                MessageBox.Show($"WebView2 initialization failed: {e.InitializationException.Message}");
            }
        }

        /// <summary>
        /// Handles the "Ask" button click event to send the user's prompt to the AI.
        /// </summary>
        private async void Ask_Click(object sender, EventArgs e)
        {
            // 1. Pre-flight checks.
            if (!IsReadyForRequest())
            {
                return;
            }

            // 2. Prepare the UI and conversation state.
            tabControl1.SelectedIndex = 0; // Switch to the Chat tab.
            await InitializeHtmlShellIfNeeded();
            await EchoUserInputToChat();
            await _webView.CoreWebView2.ExecuteScriptAsync("window.addWaitIndicator();");

            // 3. Configure the AI request.
            _aiClient.Model = (string)cmbModel.SelectedItem;
            _aiClient.InputAddText(txtInput.Text.Trim());
            txtLastError.Text = "";

            _sbMarkdown.Clear();
            _sbAccumulatedMarkdown.Clear();

            // 4. Execute the streaming request and update UI upon completion.
            await StartStreamingAsk();

            // After the response is complete, fetch the final HTML and display it in the source tab.
            await UpdateHtmlSourceView();

            // 5. Update the JSON view of the conversation.
            Chilkat.JsonObject convoJson = new Chilkat.JsonObject();
            _aiClient.ExportConvo("test",convoJson);
            DisplayJsonInTreeView(convoJson);

            _sbAccumulatedMarkdown.ToCRLF();
            textBoxMd.Text = _sbAccumulatedMarkdown.GetAsString();
        }

        /// <summary>
        /// Handles the selection change for the AI provider ComboBox.
        /// </summary>
        private void Provider_SelectedIndexChanged(object sender, EventArgs e)
        {
            string provider = (string)cmbProv.SelectedItem;
            UpdateProvider(provider);
        }

        /// <summary>
        /// Handles the "Clear" button click event to start a new conversation.
        /// </summary>
        private async void Clear_Click(object sender, EventArgs e)
        {
            _isNewConversationNeeded = true;
            await InitializeHtmlShellIfNeeded();
            textBoxMd.Clear();

            treeViewJson.BeginUpdate();
            treeViewJson.Nodes.Clear();
            treeViewJson.EndUpdate();

            txtHtml.Clear();
        }

        /// <summary>
        /// Handles the "Save Secrets" button click event.
        /// </summary>
        private void btnSaveSecrets_Click(object sender, EventArgs e)
        {
            // The "local_manager" location corresponds to the Windows Credentials Manager or macOS Keychain.
            Chilkat.Secrets secrets = new Chilkat.Secrets { Location = "local_manager" };

            // Save all secrets sequentially.
            string apiKeyName = "api_key";

            bool success = SaveSecret("Chilkat", "unlock_code", textBoxUnlockCode);
            if (success)
            {
                UnlockChilkat(textBoxUnlockCode.Text.Trim(), true);
            }
            success = SaveSecret("OpenAI", apiKeyName, textBoxOpenAI) &&
                           SaveSecret("Google", apiKeyName, textBoxGoogle) &&
                           SaveSecret("Claude", apiKeyName, textBoxClaude) &&
                           SaveSecret("xAI", apiKeyName, textBoxXAI) &&
                           SaveSecret("Perplexity", apiKeyName, textBoxPerplex) &&
                           SaveSecret("Deepseek", apiKeyName, textBoxDeepSeek) &&
                           SaveSecret("Azure", apiKeyName, textBoxAzure) &&
                           SaveSecret("Azure", "foundry_subscription", textBoxSubscription) &&
                           SaveSecret("Azure", "foundry_location", textBoxLocation) &&
                           SaveSecret("Azure", "client_id", textBoxClientId) &&
                           SaveSecret("Azure", "client_secret", textBoxClientSecret) &&
                           SaveSecret("Azure", "tenant_id", textBoxTenantId);
                           


            if (success)
            {
                // If there are no models in the models combo box, then perhaps the needed secret was provided.
                // Try getting the models..
                if (cmbModel.Items.Count == 0)
                {
                    UpdateProvider(cmbProv.Text);
                }
                MessageBox.Show("Secrets saved.");
            }
        }

        #endregion

        #region Chilkat and AI Logic

        /// <summary>
        /// Unlocks the Chilkat library with a given unlock code.
        /// </summary>
        /// <param name="unlockCode">The bundle unlock code.</param>
        /// <returns>True if successful, otherwise false.</returns>
        private bool UnlockChilkat(string unlockCode, bool showMsgBox)
        {
            Chilkat.Global glob = new Chilkat.Global();
            if (glob.UnlockStatus != 0) return true; // Already unlocked.

            if (glob.UnlockBundle(unlockCode))
            {
                return true;
            }

            if (showMsgBox) MessageBox.Show("Chilkat unlock failed. Please check your unlock code or use the trial.", "Unlock Failed");
            txtLastError.Text = "Chilkat unlock failed:\r\n" + glob.LastErrorText;
            tabControl1.SelectedIndex = 2; // Switch to the "Last Error" tab.
            return false;
        }

        /// <summary>
        /// Verifies that the Chilkat library is unlocked before making a request.
        /// </summary>
        /// <returns>True if unlocked, otherwise false.</returns>
        private bool IsChilkatUnlocked()
        {
            Chilkat.Global glob = new Chilkat.Global();
            if (glob.UnlockStatus == 0) // 0 means not unlocked
            {
                string message = "Chilkat is not unlocked. Please provide a valid unlock code on the Secrets tab.";
                MessageBox.Show(message, "Chilkat Not Unlocked");
                txtLastError.Text = message;
                tabControl1.SelectedIndex = 3; // Switch to the "Secrets" tab.
                textBoxUnlockCode.Focus();
                return false;
            }
            return true;
        }

        /// <summary>
        /// Checks if the application is ready to send an AI request by verifying Chilkat unlock status and API key presence.
        /// </summary>
        /// <returns>True if ready, otherwise false.</returns>
        private bool IsReadyForRequest()
        {
            if (!IsChilkatUnlocked()) return false;

            string provider = (string)cmbProv.SelectedItem;
            if (SetApiKey(provider))
            {
                return true;
            }

            MessageBox.Show($"API key is missing for provider: {provider}. Please add it on the Secrets tab.", "API Key Missing");
            tabControl1.SelectedIndex = 3; // Switch to the "Secrets" tab.
            return false;
        }

        /// <summary>
        /// Sets the API key for the specified AI provider in the Chilkat AI component.
        /// </summary>
        /// <param name="provider">The name of the AI provider (e.g., "OpenAI").</param>
        /// <returns>True if an API key was found and set, otherwise false.</returns>
        private bool SetApiKey(string provider)
        {
            string apiKey = null;

            switch (provider)
            {
                case "OpenAI":
                    apiKey = textBoxOpenAI.Text.Trim();
                    break;
                case "Google":
                    apiKey = textBoxGoogle.Text.Trim();
                    break;
                case "Claude":
                    apiKey = textBoxClaude.Text.Trim();
                    break;
                case "xAI":
                    apiKey = textBoxXAI.Text.Trim();
                    break;
                case "Perplexity":
                    apiKey = textBoxPerplex.Text.Trim();
                    break;
                case "Deepseek":
                    apiKey = textBoxDeepSeek.Text.Trim();
                    break;
                case "Azure":
                    apiKey = textBoxAzure.Text.Trim();
                    break;
                // The 'default' case handles any provider not listed above.
                default:
                    apiKey = null;
                    break;
            }

            if (string.IsNullOrEmpty(apiKey))
            {
                return false;
            }

            _aiClient.ApiKey = apiKey;
            return true;
        }

        /// <summary>
        /// Updates the AI provider and populates the model ComboBox with available models for that provider.
        /// </summary>
        /// <param name="provider">The name of the AI provider.</param>
        /// <returns>True if the provider was updated and models were fetched successfully, otherwise false.</returns>
        private bool UpdateProvider(string provider)
        {
            if (string.IsNullOrEmpty(provider))
            {
                // Iterate over providers and try setting the ApiKey for each.
                // Choose the 1st that succeeds.  If none succeed, then choose OpenAI and do not populate the models combo box.
                bool success = false;
                for (int i = 0; i < cmbProv.Items.Count; i++)
                {
                    var item = cmbProv.Items[i];
                    // If the items are simple strings
                    provider = item.ToString();
                    cmbProv.SelectedIndex = i;
                    success = SetApiKey(provider);
                    if (success) break;
                }

                // If successful, fall through and get the models.
                if (!success)
                {
                    cmbModel.Items.Clear();
                    cmbModel.Text = "";
                    cmbProv.SelectedIndex = 0;
                    return false;
                }

               
            }
            else if (!SetApiKey(provider))
            {
                // It's okay if the API key isn't set yet, just don't fetch models.
                // Clear the combo box of models.
                cmbModel.Items.Clear();
                cmbModel.Text = "";
                return false;
            }

            if (provider.Equals("Azure"))
            {
                _aiClient.Provider = "custom";
                _aiClient.BaseUrl = "https://chilkat.cognitiveservices.azure.com/openai/deployments/{deployment}/chat/completions?api-version=2025-01-01-preview";
                _aiClient.ApiSpec = "ChatCompletions";

                Chilkat.JsonObject azureParams = new Chilkat.JsonObject();
                azureParams.UpdateString("azure.tenantId", textBoxTenantId.Text.Trim());
                azureParams.UpdateString("azure.clientId", textBoxClientId.Text.Trim());
                azureParams.UpdateString("azure.clientSecret", textBoxClientSecret.Text.Trim());
                azureParams.UpdateString("azure.subscription", textBoxSubscription.Text.Trim());
                azureParams.UpdateString("azure.location", textBoxLocation.Text.Trim());

                if (textBoxTenantId.Text.Trim().Length > 0 &&
                    textBoxClientId.Text.Trim().Length > 0 &&
                    textBoxClientSecret.Text.Trim().Length > 0 &&
                    textBoxSubscription.Text.Trim().Length > 0 &&
                    textBoxLocation.Text.Trim().Length > 0)
                {
                    _aiClient.SetProviderParams(azureParams);
                }
                else
                {
                    MessageBox.Show("Please provide all required Azure Foundry parameters on the Secrets tab.", "Azure Parameters Missing");
                    return false;
                }
            }
            else
            {
                _aiClient.Provider = provider;
            }
               

            Chilkat.StringTable models = new Chilkat.StringTable();
            if (!_aiClient.GetModels(models))
            {
                MessageBox.Show(_aiClient.LastErrorText, "Failed to Get Models");
                return false;
            }

            // Populate the models dropdown.
            cmbModel.Items.Clear();
            for (int i = 0; i < models.Count; i++)
            {
                cmbModel.Items.Add(models.StringAt(i));
            }

            // Auto-select a preferred model if available.
            var preferredModels = new List<string>
            {
                "gpt-4o", "claude-opus-4-1-20250805", "gemini-2.5-flash",
                "deepseek-chat", "sonar", "grok-4", "gpt-4o"
            };

            string bestModel = preferredModels.FirstOrDefault(m => cmbModel.Items.Contains(m));

            if (bestModel != null)
            {
                cmbModel.SelectedItem = bestModel;
            }
            else if (cmbModel.Items.Count > 0)
            {
                cmbModel.SelectedIndex = 0;
            }

            return true;
        }

        #endregion

        #region Streaming Response Handling

        /// <summary>
        /// Initiates the asynchronous AI request and starts polling for streaming responses.
        /// </summary>
        private async System.Threading.Tasks.Task StartStreamingAsk()
        {
            _aiClient.Streaming = true;

            // Run the synchronous, blocking `Ask` method on a background thread to keep the UI responsive.
            bool success = await System.Threading.Tasks.Task.Run(() => _aiClient.Ask("text"));

            if (!success)
            {
                txtLastError.Text = _aiClient.LastErrorText;
                tabControl1.SelectedIndex = 1; // Switch to the "Last Error" tab.
                return;
            }

            // Once the request is initiated, remove the wait indicator.
            await _webView.CoreWebView2.ExecuteScriptAsync("window.removeWaitIndicator();");

            // Start the background task that will periodically check for new data chunks.
            _cancellationTokenSource = new CancellationTokenSource();
            await RunPeriodicPollingAsync(TimeSpan.FromMilliseconds(100));
        }

        /// <summary>
        /// Runs a loop that polls the Chilkat AI component for streaming data chunks at a specified interval.
        /// </summary>
        /// <param name="interval">The time to wait between polls if no data is available.</param>
        private async System.Threading.Tasks.Task RunPeriodicPollingAsync(TimeSpan interval)
        {
            while (!_cancellationTokenSource.IsCancellationRequested)
            {
                bool receivedData = await PollForAiDeltas();

                // If no data was received, wait for the specified interval before polling again.
                if (!receivedData)
                {
                    await System.Threading.Tasks.Task.Delay(interval);
                }
            }
        }

        /// <summary>
        /// Polls for the next available chunk of the AI's response and processes it.
        /// </summary>
        /// <returns>True if data was received and processed, false if no new data was available.</returns>
        private async Task<bool> PollForAiDeltas()
        {
            // PollAi is non-blocking and quickly checks if data is available.
            int pollResult = _aiClient.PollAi(false);

            if (pollResult == 0) return false; // No new output is available.
            if (pollResult != 1)
            {
                // An error occurred during polling.
                _cancellationTokenSource.Cancel();
                txtLastError.Text += "\r\nPolling Error: " + _aiClient.LastErrorText;
                tabControl1.SelectedIndex = 1;
                return false;
            }

            // Data is available, so retrieve the next event.
            if (!_aiClient.NextAiEvent(5000, _sbEventName, _sbDelta))
            {
                _cancellationTokenSource.Cancel();
                txtLastError.Text += "\r\nNextAiEvent Error: " + _aiClient.LastErrorText;
                tabControl1.SelectedIndex = 1;
                return false;
            }

            // Process the event based on its name.
            string eventName = _sbEventName.GetAsString();
            switch (eventName)
            {
                case "delta":
                    await ProcessDeltaEvent();
                    break;
                case "null_terminator":
                    await ProcessEndOfStreamEvent();
                    break;
                case "empty":
                    // Some providers send empty events; ignore them and continue polling immediately.
                    break;
            }

            return true;
        }

        /// <summary>
        /// Processes a "delta" event, which contains a new chunk of text from the AI.
        /// </summary>
        private async System.Threading.Tasks.Task ProcessDeltaEvent()
        {
            // Accumulate the markdown and convert it to JavaScript calls for the WebView.
            _sbMarkdown.AppendSb(_sbDelta);
            _sbAccumulatedMarkdown.AppendSb(_sbDelta);  
            _sbJsCalls.Clear();
            _sbMarkdown.MarkdownToHtml(_streamingOptions, _sbJsCalls);

            if (_sbJsCalls.Length > 0)
            {
                await _webView.CoreWebView2.ExecuteScriptAsync(_sbJsCalls.GetAsString());
            }
        }

        /// <summary>
        /// Processes the "null_terminator" event, which signals the end of the AI's response.
        /// </summary>
        private async System.Threading.Tasks.Task ProcessEndOfStreamEvent()
        {
            // Finalize the HTML output by flushing any remaining content.
            _streamingOptions.UpdateBool("streamingFinished", true);
            _sbJsCalls.Clear();
            _sbMarkdown.MarkdownToHtml(_streamingOptions, _sbJsCalls);

            if (_sbJsCalls.Length > 0)
            {
                await _webView.CoreWebView2.ExecuteScriptAsync(_sbJsCalls.GetAsString());
            }

            // Clean up the option and stop the polling task.
            _streamingOptions.Delete("streamingFinished");
            _cancellationTokenSource.Cancel();
        }

        #endregion

        #region WebView2 and HTML Management

        /// <summary>
        /// If a new conversation is needed, this method creates and loads a new empty HTML shell into the WebView2 control.
        /// </summary>
        private async System.Threading.Tasks.Task InitializeHtmlShellIfNeeded()
        {
            if (!_isNewConversationNeeded) return;

            // Start a new conversation named "test". Delete any existing one.
            _aiClient.DeleteConvo("test");
            _aiClient.NewConvo("test", "", "");

            // Configure the HTML shell options.
            Chilkat.JsonObject htmlOptions = new Chilkat.JsonObject();
            SetHtmlThemeOptions(htmlOptions);
            htmlOptions.UpdateBool("streamingShell", true);

            // Configure streaming options to match Prism usage.
            _streamingOptions.UpdateBool("usesPrism", cmbPrism.Text != "none");

            // Generate the HTML shell.
            Chilkat.StringBuilder sbHtml = new Chilkat.StringBuilder();
            Chilkat.StringBuilder sbMarkdown = new Chilkat.StringBuilder(); // Empty builder for this call
            sbMarkdown.MarkdownToHtml(htmlOptions, sbHtml);

            txtHtml.Text = sbHtml.GetAsString();

            // Load the generated HTML into the WebView2 control.
            await NavigateToStringAndWaitAsync(sbHtml.GetAsString());

            _isNewConversationNeeded = false;
        }

        /// <summary>
        /// Configures the JSON options object for generating the HTML shell based on UI selections.
        /// </summary>
        /// <param name="jsonOptions">The JSON object to configure.</param>
        private void SetHtmlThemeOptions(Chilkat.JsonObject jsonOptions)
        {
            // Set the overall HTML theme.
            string theme = cmbHtmlStyles.Text;
            if (theme != "none")
            {
                jsonOptions.UpdateString("theme", theme);
            }

            if (theme == "ChatGPT")
            {
                jsonOptions.UpdateString("ChatGPT.max-width", "80ch");
            }

            // Set the Prism theme for syntax highlighting.
            string prismTheme = cmbPrism.Text;
            if (prismTheme == "none")
            {
                jsonOptions.Delete("usesPrism");
            }
            else
            {
                jsonOptions.UpdateBool("usesPrism", true);
                jsonOptions.UpdateString("prism.theme", prismTheme);
                jsonOptions.UpdateString("prism.version", "1.29.0"); // Optionally specify version
            }
        }

        /// <summary>
        /// Displays the user's input in the chat window for context.
        /// </summary>
        private async System.Threading.Tasks.Task EchoUserInputToChat()
        {
            string encodedInput = EncodeBasicHtmlEntities(txtInput.Text.Trim());
            string html = $"<div style=\"display: flex;\"><p style=\"margin-left: auto;\"><code>{encodedInput}</code></p></div>\n";
            await AppendHtmlToSelectorAsync("div", html);
        }

        /// <summary>
        /// Encodes basic HTML entities to prevent rendering issues when displaying user input.
        /// </summary>
        /// <param name="input">The raw string.</param>
        /// <returns>The HTML-encoded string.</returns>
        public static string EncodeBasicHtmlEntities(string input)
        {
            if (string.IsNullOrEmpty(input))
                return input;

            return new System.Text.StringBuilder(input)
                .Replace("&", "&amp;")
                .Replace("<", "&lt;")
                .Replace(">", "&gt;")
                .Replace("\"", "&quot;")
                .Replace("'", "&apos;")
                .ToString();
        }

        /// <summary>
        /// Navigates the WebView2 control to a given HTML string and waits for the content to be fully loaded.
        /// </summary>
        /// <param name="htmlContent">The HTML content to display.</param>
        private System.Threading.Tasks.Task NavigateToStringAndWaitAsync(string htmlContent)
        {
            var tcs = new TaskCompletionSource<bool>();

            EventHandler<CoreWebView2NavigationCompletedEventArgs> handler = null;
            handler = (sender, args) =>
            {
                _webView.NavigationCompleted -= handler; // Unsubscribe to prevent memory leaks.
                tcs.SetResult(true);
            };

            _webView.NavigationCompleted += handler;
            _webView.CoreWebView2.NavigateToString(htmlContent);

            return tcs.Task;
        }

        /// <summary>
        /// Executes a JavaScript function in the WebView2 control to append HTML to a specified element.
        /// </summary>
        /// <param name="selector">The CSS selector of the parent element.</param>
        /// <param name="htmlContent">The HTML string to append.</param>
        private async System.Threading.Tasks.Task AppendHtmlToSelectorAsync(string selector, string htmlContent)
        {
            if (_webView?.CoreWebView2 != null)
            {
                // Serialize arguments to ensure they are correctly formatted as JavaScript string literals.
                string escapedSelector = JsonSerializer.Serialize(selector);
                string escapedHtml = JsonSerializer.Serialize(htmlContent);
                string script = $"window.appendHtmlBySelector({escapedSelector}, {escapedHtml});";
                await _webView.CoreWebView2.ExecuteScriptAsync(script);
            }
        }

        /// <summary>
        /// Retrieves the full HTML source from the WebView2 control.
        /// </summary>
        /// <returns>A string containing the current HTML of the page.</returns>
        private async Task<string> GetWebViewHtmlAsync()
        {
            if (_webView?.CoreWebView2 == null)
            {
                return null;
            }

            // Execute a script to get the document's full outer HTML.
            string htmlJson = await _webView.CoreWebView2.ExecuteScriptAsync("document.documentElement.outerHTML;");

            // The result is a JSON-encoded string. Deserialize it to get the raw HTML.
            return htmlJson == null || htmlJson == "null" ? "" : JsonSerializer.Deserialize<string>(htmlJson);
        }



        /// <summary>
        /// Fetches the current HTML from the WebView and displays it in the HTML source text box.
        /// </summary>
        private async System.Threading.Tasks.Task UpdateHtmlSourceView()
        {
            string currentHtml = await GetWebViewHtmlAsync();
            if (currentHtml != null)
            {
                // Use Chilkat StringBuilder for potentially large HTML content and proper line endings.
                Chilkat.StringBuilder sb = new Chilkat.StringBuilder();
                sb.Append(currentHtml);
                sb.ToCRLF();
                txtHtml.Text = sb.GetAsString();
            }
            else
            {
                MessageBox.Show("Could not retrieve HTML. WebView2 may not be initialized.", "Error");
            }
        }

        #endregion

        #region Secrets Management

        /// <summary>
        /// Loads all secrets (API keys and unlock code) from the local secrets manager.
        /// </summary>
        private void CheckLoadSecrets()
        {
            // The Chilkat unlock code can be retrieved even when the library is in trial mode.
            LoadSecret("Chilkat", "unlock_code", textBoxUnlockCode);

            // If an unlock code was found, try to unlock the library now.
            Chilkat.Global glob = new Chilkat.Global();
            if (glob.UnlockStatus == 0 && !string.IsNullOrWhiteSpace(textBoxUnlockCode.Text))
            {
                if (!UnlockChilkat(textBoxUnlockCode.Text.Trim(), false)) return;
            }

            // Loading other secrets requires Chilkat to be unlocked.
            string name = "api_key";
            LoadSecret("OpenAI", name, textBoxOpenAI);
            LoadSecret("Google", name, textBoxGoogle);
            LoadSecret("Claude", name, textBoxClaude);
            LoadSecret("xAI", name, textBoxXAI);
            LoadSecret("Perplexity", name, textBoxPerplex);
            LoadSecret("Deepseek", name, textBoxDeepSeek);
            LoadSecret("Azure", name, textBoxAzure);
            LoadSecret("Azure", "foundry_subscription", textBoxSubscription);
            LoadSecret("Azure", "foundry_location", textBoxLocation);
            LoadSecret("Azure", "client_id", textBoxClientId);
            LoadSecret("Azure", "client_secret", textBoxClientSecret);
            LoadSecret("Azure", "tenant_id", textBoxTenantId);

        }

        /// <summary>
        /// Loads a single secret for a given provider and populates a TextBox.
        /// </summary>
        /// <param name="provider">The service name (e.g., "OpenAI", "Chilkat").</param>
        /// <param name="textBox">The TextBox to populate with the secret.</param>
        private void LoadSecret(string provider, string name, TextBox textBox)
        {
            Chilkat.JsonObject json = new Chilkat.JsonObject();
            json.UpdateString("appName", "TestAiApp");
            json.UpdateString("service", provider);
            json.UpdateString("username", name);

            Chilkat.Secrets secrets = new Chilkat.Secrets();
            string secret = secrets.GetSecretStr(json);
            if (!string.IsNullOrEmpty(secret))
            {
                textBox.Text = secret;
            }
        }

        /// <summary>
        /// Saves a single secret to the local secrets manager.
        /// </summary>
        /// <param name="provider">The service name (e.g., "OpenAI", "Chilkat").</param>
        /// <param name="secret">The secret value to save.</param>
        /// <returns>True if successful, otherwise false.</returns>
        private bool SaveSecret(string provider, string name, System.Windows.Forms.TextBox textBox)
        {
            string secret = textBox.Text;
            secret = secret.Trim();
            if (string.IsNullOrEmpty(secret)) return true; // Nothing to save.

            Chilkat.JsonObject json = new Chilkat.JsonObject();
            json.UpdateString("appName", "TestAiApp");
            json.UpdateString("service", provider);
            json.UpdateString("username", name);

            Chilkat.Secrets secrets = new Chilkat.Secrets();

            if (secret.Equals("delete"))
            {
                secrets.DeleteSecret(json);
                textBox.Text = "";
            }
            else if (!secrets.UpdateSecretStr(json, secret))
            {
                string errorMessage = "Failed to save secret: " + secrets.LastErrorText;
                MessageBox.Show(errorMessage, "Save Secret Failed");
                txtLastError.Text = errorMessage;
                tabControl1.SelectedIndex = 2; // Switch to "Last Error" tab.
                return false;
            }

            return true;
        }

        #endregion


        #region TreeView Management

        private void DisplayJsonInTreeView(Chilkat.JsonObject json)
        {
            treeViewJson.BeginUpdate();
            treeViewJson.Nodes.Clear();

            AddObjectNodes(treeViewJson.Nodes, json);

            treeViewJson.ExpandAll();
            treeViewJson.EndUpdate();

        }

        private void AddObjectNodes(TreeNodeCollection nodes, JsonObject obj)
        {
            int count = obj.Size;

            for (int i = 0; i < count; i++)
            {
                string name = obj.NameAt(i);
                int type = obj.TypeAt(i);   // 1=string,2=number,3=object,4=array,5=bool,6=null

                TreeNode node;

                switch (type)
                {
                    case 3: // object
                        {
                            JsonObject child = obj.ObjectAt(i);
                            node = nodes.Add(name);
                            AddObjectNodes(node.Nodes, child);
                            break;
                        }

                    case 4: // array
                        {
                            JsonArray arr = obj.ArrayAt(i);
                            node = nodes.Add(name + " []");
                            AddArrayNodes(node, arr);
                            break;
                        }

                    case 1: // string
                        node = nodes.Add($"{name}: \"{obj.StringAt(i)}\"");
                        break;

                    case 2: // number
                        node = nodes.Add($"{name}: {obj.StringAt(i)}");
                        break;

                    case 5: // bool
                        node = nodes.Add($"{name}: {obj.BoolAt(i).ToString().ToLowerInvariant()}");
                        break;

                    case 6: // null
                        node = nodes.Add($"{name}: null");
                        break;

                    default:
                        node = nodes.Add($"{name}: <unknown>");
                        break;
                }
            }
        }

        private void AddArrayNodes(TreeNode parentNode, JsonArray arr)
        {
            if (arr == null) return;

            int count = arr.Size;

            for (int i = 0; i < count; i++)
            {
                int type = arr.TypeAt(i);   // 1=string,2=number,3=object,4=array,5=bool,6=null
                TreeNode node;

                switch (type)
                {
                    case 3: // object
                        {
                            JsonObject childObj = arr.ObjectAt(i);
                            node = parentNode.Nodes.Add($"[{i}]");
                            AddObjectNodes(node.Nodes, childObj);
                            break;
                        }

                    case 4: // array
                        {
                            JsonArray childArr = arr.ArrayAt(i);
                            node = parentNode.Nodes.Add($"[{i}] []");
                            AddArrayNodes(node, childArr);
                            break;
                        }

                    case 1: // string
                        node = parentNode.Nodes.Add($"[{i}]: \"{arr.StringAt(i)}\"");
                        break;

                    case 2: // number
                        node = parentNode.Nodes.Add($"[{i}]: {arr.StringAt(i)}");
                        break;

                    case 5: // bool
                        node = parentNode.Nodes.Add($"[{i}]: {arr.BoolAt(i).ToString().ToLowerInvariant()}");
                        break;

                    case 6: // null
                        node = parentNode.Nodes.Add($"[{i}]: null");
                        break;

                    default:
                        node = parentNode.Nodes.Add($"[{i}]: <unknown>");
                        break;
                }
            }
        }

        #endregion

    }
}